import Foundation
import SwiftUI

/// Service responsible for managing user privacy and data operations
class PrivacyService: ObservableObject {
    static let shared = PrivacyService()
    
    @Published var isDeletingData = false
    @Published var isExportingData = false
    
    private init() {}
    
    // MARK: - Data Deletion
    
    /// Deletes all user data from the app
    func deleteAllUserData() async throws {
        await MainActor.run {
            isDeletingData = true
        }
        
        defer {
            Task { @MainActor in
                isDeletingData = false
            }
        }
        
        try await withThrowingTaskGroup(of: Void.self) { group in
            // Clear UserDefaults
            group.addTask {
                await self.clearUserDefaults()
            }
            
            // Clear Core Data if available
            group.addTask {
                await self.clearCoreData()
            }
            
            // Clear HealthKit cache
            group.addTask {
                await self.clearHealthKitCache()
            }
            
            // Clear file system cache
            group.addTask {
                await self.clearFileSystemCache()
            }
            
            try await group.waitForAll()
        }
    }
    
    private func clearUserDefaults() async {
        let defaults = UserDefaults.standard
        let keysToRemove = [
            "hasAcceptedDisclaimer",
            "disclaimerAcceptedDate",
            "lastPredictionDate",
            "userBirthDate",
            "userGender",
            "healthKitAuthorized",
            "predictionHistory",
            "userPreferences",
            "surveyResponses"
        ]
        
        keysToRemove.forEach { key in
            defaults.removeObject(forKey: key)
        }
        
        defaults.synchronize()
    }
    
    private func clearCoreData() async {
        // Placeholder for Core Data cleanup
        // This would integrate with PersistenceController when available
        // try await PersistenceController.shared.deleteAllData()
    }
    
    private func clearHealthKitCache() async {
        // Clear any cached HealthKit data
        // This would integrate with HealthKitService when available
        // await HealthKitService.shared.clearCache()
    }
    
    private func clearFileSystemCache() async {
        // Clear temporary files and caches
        let fileManager = FileManager.default
        let cacheURL = fileManager.urls(for: .cachesDirectory, in: .userDomainMask).first
        
        if let cacheURL = cacheURL {
            do {
                let contents = try fileManager.contentsOfDirectory(at: cacheURL, includingPropertiesForKeys: nil)
                for url in contents {
                    try? fileManager.removeItem(at: url)
                }
            } catch {
                // Handle file system errors gracefully
            }
        }
    }
    
    // MARK: - Data Export
    
    /// Generates a comprehensive data export
    func generateDataExport() async -> String {
        await MainActor.run {
            isExportingData = true
        }
        
        defer {
            Task { @MainActor in
                isExportingData = false
            }
        }
        
        var export = "Death App - Personal Data Export\n"
        export += "Generated: \(formatDate(Date()))\n"
        export += "Export Version: 1.0\n\n"
        
        // User Settings
        export += await generateSettingsExport()
        
        // Health Data Summary
        export += await generateHealthDataExport()
        
        // Prediction History
        export += await generatePredictionHistoryExport()
        
        // Survey Responses
        export += await generateSurveyResponsesExport()
        
        // Privacy Information
        export += generatePrivacyInformation()
        
        return export
    }
    
    private func generateSettingsExport() async -> String {
        var section = "=== USER SETTINGS ===\n"
        let defaults = UserDefaults.standard
        
        section += "Disclaimer Accepted: \(defaults.bool(forKey: "hasAcceptedDisclaimer"))\n"
        
        if let date = defaults.object(forKey: "disclaimerAcceptedDate") as? Date {
            section += "Disclaimer Date: \(formatDate(date))\n"
        }
        
        if let birthDate = defaults.object(forKey: "userBirthDate") as? Date {
            section += "Birth Date: \(formatDate(birthDate))\n"
        }
        
        if let gender = defaults.string(forKey: "userGender") {
            section += "Gender: \(gender)\n"
        }
        
        section += "HealthKit Authorized: \(defaults.bool(forKey: "healthKitAuthorized"))\n"
        section += "\n"
        
        return section
    }
    
    private func generateHealthDataExport() async -> String {
        var section = "=== HEALTH DATA SUMMARY ===\n"
        section += "Note: Raw health data is managed by HealthKit and remains secure.\n"
        section += "Only aggregated summaries are included in this export.\n\n"
        
        // Placeholder for health data integration
        section += "Health Data Types Accessed:\n"
        section += "• Heart Rate (if authorized)\n"
        section += "• Step Count (if authorized)\n"
        section += "• Sleep Analysis (if authorized)\n"
        section += "• Active Energy (if authorized)\n\n"
        
        return section
    }
    
    private func generatePredictionHistoryExport() async -> String {
        var section = "=== PREDICTION HISTORY ===\n"
        
        // Placeholder for prediction history integration
        if let historyData = UserDefaults.standard.data(forKey: "predictionHistory") {
            section += "Prediction history available: \(historyData.count) bytes\n"
            section += "Note: Detailed prediction data requires database integration.\n"
        } else {
            section += "No prediction history found.\n"
        }
        
        section += "\n"
        return section
    }
    
    private func generateSurveyResponsesExport() async -> String {
        var section = "=== SURVEY RESPONSES ===\n"
        
        // Placeholder for survey data integration
        if let surveyData = UserDefaults.standard.data(forKey: "surveyResponses") {
            section += "Survey responses available: \(surveyData.count) bytes\n"
            section += "Note: Detailed survey data requires database integration.\n"
        } else {
            section += "No survey responses found.\n"
        }
        
        section += "\n"
        return section
    }
    
    private func generatePrivacyInformation() -> String {
        var section = "=== PRIVACY INFORMATION ===\n"
        section += "Data Storage: All data is stored locally on your device\n"
        section += "Data Sharing: No personal data is shared with external services\n"
        section += "Data Retention: Data is retained until manually deleted\n"
        section += "Your Rights: You can export or delete your data at any time\n\n"
        
        section += "=== DATA CATEGORIES ===\n"
        section += "1. Identity Data: None collected\n"
        section += "2. Health Data: Stored via HealthKit (secure)\n"
        section += "3. Usage Data: App interactions and preferences\n"
        section += "4. Diagnostic Data: Error logs (anonymous)\n\n"
        
        return section
    }
    
    // MARK: - Data Retention Settings
    
    /// Available data retention periods
    enum RetentionPeriod: String, CaseIterable {
        case oneMonth = "1_month"
        case threeMonths = "3_months"
        case sixMonths = "6_months"
        case oneYear = "1_year"
        case indefinite = "indefinite"
        
        var displayName: String {
            switch self {
            case .oneMonth: return "1 Month"
            case .threeMonths: return "3 Months"
            case .sixMonths: return "6 Months"
            case .oneYear: return "1 Year"
            case .indefinite: return "Keep Forever"
            }
        }
        
        var days: Int? {
            switch self {
            case .oneMonth: return 30
            case .threeMonths: return 90
            case .sixMonths: return 180
            case .oneYear: return 365
            case .indefinite: return nil
            }
        }
    }
    
    /// Get current data retention setting
    var currentRetentionPeriod: RetentionPeriod {
        get {
            let rawValue = UserDefaults.standard.string(forKey: "dataRetentionPeriod") ?? RetentionPeriod.indefinite.rawValue
            return RetentionPeriod(rawValue: rawValue) ?? .indefinite
        }
        set {
            UserDefaults.standard.set(newValue.rawValue, forKey: "dataRetentionPeriod")
            scheduleDataCleanup(for: newValue)
        }
    }
    
    /// Schedule automatic data cleanup based on retention period
    private func scheduleDataCleanup(for period: RetentionPeriod) {
        // Implementation would integrate with background task scheduling
        // For now, just store the preference
        UserDefaults.standard.set(Date(), forKey: "dataRetentionPeriodSet")
    }
    
    // MARK: - Utility Functions
    
    private func formatDate(_ date: Date) -> String {
        let formatter = DateFormatter()
        formatter.dateStyle = .medium
        formatter.timeStyle = .medium
        return formatter.string(from: date)
    }
}

// MARK: - Preview Support
#if DEBUG
extension PrivacyService {
    static let preview = PrivacyService()
}
#endif